using System.Drawing;
using System.Windows.Media.Imaging;

namespace v2rayN.Manager;

public sealed class WindowsManager
{
    private static readonly Lazy<WindowsManager> instance = new(() => new());
    public static WindowsManager Instance => instance.Value;
    private static readonly string _tag = "WindowsHandler";

    public async Task<Icon> GetNotifyIcon(Config config)
    {
        try
        {
            var index = (int)config.SystemProxyItem.SysProxyType;

            //Load from routing setting
            var createdIcon = await GetNotifyIcon4Routing(config);
            if (createdIcon != null)
            {
                return createdIcon;
            }

            //Load from local file
            var fileName = Utils.GetPath($"NotifyIcon{index + 1}.ico");
            if (File.Exists(fileName))
            {
                return new Icon(fileName);
            }
            return index switch
            {
                0 => Properties.Resources.NotifyIcon1,
                1 => Properties.Resources.NotifyIcon2,
                2 => Properties.Resources.NotifyIcon3,
                3 => Properties.Resources.NotifyIcon4,
                _ => Properties.Resources.NotifyIcon1, // default
            };
        }
        catch (Exception ex)
        {
            Logging.SaveLog(_tag, ex);
            return Properties.Resources.NotifyIcon1;
        }
    }

    public System.Windows.Media.ImageSource GetAppIcon(Config config)
    {
        var index = (int)config.SystemProxyItem.SysProxyType + 1;
        return BitmapFrame.Create(new Uri($"pack://application:,,,/Resources/NotifyIcon{index}.ico", UriKind.RelativeOrAbsolute));
    }

    private async Task<Icon?> GetNotifyIcon4Routing(Config config)
    {
        try
        {
            var item = await ConfigHandler.GetDefaultRouting(config);
            if (item == null || item.CustomIcon.IsNullOrEmpty() || !File.Exists(item.CustomIcon))
            {
                return null;
            }

            var color = ColorTranslator.FromHtml("#3399CC");
            var index = (int)config.SystemProxyItem.SysProxyType;
            if (index > 0)
            {
                color = (new[] { Color.Red, Color.Purple, Color.DarkGreen, Color.Orange, Color.DarkSlateBlue, Color.RoyalBlue })[index - 1];
            }

            var width = 128;
            var height = 128;

            Bitmap bitmap = new(width, height);
            var graphics = Graphics.FromImage(bitmap);
            SolidBrush drawBrush = new(color);

            graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
            //graphics.FillRectangle(drawBrush, new Rectangle(0, 0, width, height));
            graphics.DrawImage(new Bitmap(item.CustomIcon), 0, 0, width, height);
            graphics.FillEllipse(drawBrush, width / 2, width / 2, width / 2, width / 2);

            var createdIcon = Icon.FromHandle(bitmap.GetHicon());

            drawBrush.Dispose();
            graphics.Dispose();
            bitmap.Dispose();

            return createdIcon;
        }
        catch (Exception ex)
        {
            Logging.SaveLog(_tag, ex);
            return null;
        }
    }

    public void RegisterGlobalHotkey(Config config, Action<EGlobalHotkey> handler, Action<bool, string>? update)
    {
        HotkeyManager.Instance.UpdateViewEvent += update;
        HotkeyManager.Instance.HotkeyTriggerEvent += handler;
        HotkeyManager.Instance.Load();
    }
}
